/* Namespaces:

These two namespaces (AmpPannel and AntPannel) are accessible through ACOMTools
for the extensions - not directly

AntPannel
            .position           - antenna selection
AmpPannel
            .alert(str)         - message
            .Item
                ("ForwardPower")     
                ("ReflectedPower")   
                ("Prot")             
                ("Heater")           
                ("Plate")            
                ("Drive")            
                ("Opr")              
                ("Stby")             
                ("Comm")             
                ("Power")            
                ("Frequency")        
                ("Antenna") or ("T") or ("Misc")
                ("Status")
            .text
                ("Text1")
                ("Text2")   // Labels
                
ACOMTools   
            .AmpPannel
            .AntPannel
            .bgColor
            .fgColor
            .Core
            .Group
            .Type
            .PlaySound(x)
            .RaiseEvent(name,content)
            .RaiseUIError(text)
            .Translimt(packet)
            
Settings
            .Item(name) = string value
            .Option(name) = numeric value
            
// Core only
Translator
            .Transmit

AcomApp     - not important - placed mostly for future use

Driver      See USPC docs

extensions
            .Item(ext_name)
            .IsRunning(name)
            .Shutdown()
            .StartUp()
            .Start(name)
            .Stop(name)
*/


///////////////////////////////

// Exported functions - for the extensions
// =======================================

// Changes the antenna and segment
// return: success state
// freq - frequency reported
// nAntenna - Ant ID or:
//      -1: Leave the amplifier to choose last used
//      -2: Select first applicable (using user data)
//      -3: Use current antenna
function SelectSegAndAntenna(seg,nAntenna) {
    var i;
    for (i = 0; i < segments.length; i++) {
        if (segments[i].segment == seg) {
            return SelectSegAndAntennaObj(segments[i],nAntenna);
        }
    }
    return false;
}
function SelectSegAndAntennaObj(seg,nAntenna) {
    var p;
    if (seg == null) {
        ACOMTools.RaiseEvent("Core error","Segment not found for the freq="+freq);
        return false;
    }
    if (nAntenna >= 0 && nAntenna < 10) {
        p = Packet(g_AmpType,g_SelfGroup,"W1");
        p.param(0) = seg.segment;
        p.param(1) = nAntenna;
        Translator.Transmit(p);
        RefreshFrequencyMeter();
    } else {
        var antID;
        switch (nAntenna) {
            case -2:
                antID = GetFirstApplicableAntenna(seg);
                ACOMTools.RaiseEvent("DEBUG","Applicable antenna = " + antID);
                if (antID >= 0) {
                    p = Packet(g_AmpType,g_SelfGroup,"W1");
                    p.param(0) = seg.segment;
                    p.param(1) = antID;
                    Translator.Transmit(p);
                    RefreshFrequencyMeter();
                    break;
                }
            case -1:
                p = Packet(g_AmpType,g_SelfGroup,"W2");
                p.param(0) = seg.segment;
                Translator.Transmit(p);
                RefreshFrequencyMeter();
                break;
            case -3:
                antID = AntPannel.Position;
                if (antID >= 0 && antID < 10) {
                    p = Packet(g_AmpType,g_SelfGroup,"W1");
                    p.param(0) = seg.segment;
                    p.param(1) = antID;
                    Translator.Transmit(p);
                    RefreshFrequencyMeter();
                    break;
                }
                ACOMTools.RaiseEvent("Core error","Current antenna unknown");
                return false;
            default:
                ACOMTools.RaiseEvent("Core error","Invalid antenna parameter for the SelectFreqAndAntenna");
                return false;
        }
    }
    ACOMTools.RaiseEvent("Core report","Segment/antenna request sent");
    return true;
}
function SelectFreqAndAntenna(freq,nAntenna) {
    var seg = FindSegmentByFrequency(freq);
    return SelectSegAndAntennaObj(seg,nAntenna);
}

// Warning this function returns a script object
function FindSegmentByFrequency(freq) {
    ACOMTools.RaiseEvent("DEBUG","Freq=" + freq);
    var i;
    for (i = 0; i < segments.length; i++) {
        if (freq > segments[i].low && freq <= segments[i].high) {
            ACOMTools.RaiseEvent("DEBUG","SegInd=" + i);
            return segments[i];
        }
    }
    ACOMTools.RaiseEvent("DEBUG","NULL");
    return null; // error not found
}

// returns ID of the first applicable antenna
// defined in the user data.
// return: Antenna ID or -1 if no antenna was configured or segment does not exist
function GetFirstApplicableAntenna(sg) {
    // sg.segment is the ID for search
    ACOMTools.RaiseEvent("DEBUG","sg=" + sg.toString());
    return Antennas.FindAntennaForSegment(sg.segment);
}
function SetRangeUserMode(b) {
        ACOMTools.RaiseEvent("DBG","setting mode ... (in core 0)");
	if (!ACOMTools.IsConnected) return false;
	if ( LastSegment < 0 || LastAntenna < 0) return false;
        ACOMTools.RaiseEvent("DBG","setting mode ... (in core)");
	var p;
	if (b) {
  	        ACOMTools.RaiseEvent("DBG","setting mode - segment and antenna");
		p = DefAmpPacket("W8");
 		p.param(0) = LastSegment;
 		p.param(1) = LastAntenna;
	} else {
		p = DefAmpPacket("W5");
 		p.param(0) = LastSegment;
	}
	ACOMTools.Transmit(p);
	RefreshFrequencyMeter();
	return true;
}

// Utilities
function RefreshFrequencyMeter() {
	var p = DefAmpPacket("F");
	ACOMTools.Transmit(p);
}
function Packet(toType,toGroup,Cmd) {
    var p = new ActiveXObject("newObjects.USPC.ACOMSerialProtocolPacket");
    p.packetType = 0;
    p.command = Cmd;
    p.txType = g_SelfType;
	p.txGroup = g_SelfGroup;
	p.rxGroup = toGroup;
	p.rxType = toType;
    return p;	
}
function DefAmpPacket(Cmd) {
    var p = new ActiveXObject("newObjects.USPC.ACOMSerialProtocolPacket");
    p.packetType = 0;
    p.command = Cmd;
    p.txType = g_SelfType;
	p.txGroup = g_SelfGroup;
	p.rxGroup = g_SelfGroup;
	p.rxType = g_AmpType;
    return p;	
}
// Accessed from the config dialog
function GetPersistenceManager() {
    return Settings;
}
function GetAntPannel() {
    return AntPannel;
}
function GetFrequencyData(index,name) {
    if (index >= segments.length) return null;
    return segments[index][name];
}
function GetPosFrequencies(index) {
    // returns "1,5,8" list
    return Antennas.GetFreq(index);
}
function GetSegmentByID(id) {
    var i;
    for (i = 0; i < segments.length; i++) {
        if (segments[i].segment == id) return segments[i];
    }
    return null;
}
// Misc API
function SetAntenna(nAnt) {
    var p;
    if (nAnt >= 0 && nAnt < 10) {
        p = Packet(g_AmpType,g_SelfGroup,"W3");
        p.param(0) = nAnt;
        Translator.Transmit(p);
    }
}



// Classes
function Ant() {
    var i;
    this.antennas = new Array();
    for (i = 0; i < 10; i++) {
        this.antennas[i] = {
            frequencies:    Settings("Position" + i)    
        };
    }
}
Ant.prototype.Save = function() {
    var i;
    for (i = 0; i < this.antennas.length; i++) {
        Settings("Position" + i) = this.antennas[i];
    }
}
Ant.prototype.GetFreq = function (index) {
    if (index >= this.antennas.length) return "";
    return this.antennas[index].frequencies;
}
Ant.prototype.SetFreq = function (index,freq) {
    if (index >= this.antennas.length) return false;
    this.antennas[index].frequencies = freq;
    Settings("Position" + index) = freq;
    this.UpdateOtherAntennas(index,freq);
}
Ant.prototype.UpdateOtherAntennas = function(index,freq) {
    var arr;
    var i,j,k;
    var arrUsed = freq.split(",");
    var arrNew;
    var bSkip;
    if (arrUsed == null) return;
    if (arrUsed.length == 0) return;
    for (i = 0; i < this.antennas.length; i++) {
        if (i == index) continue;
        arr = this.antennas[i].frequencies.split(",");
        if (arr == null) continue;
        arrNew = new Array();
        for (j = 0; j < arr.length; j++) {
            bSkip = false;    
            for (k = 0; k < arrUsed.length; k++) {
                if (arr[j] == arrUsed[k]) {
                    bSkip = true;
                    break;
                } 
            }
            if (!bSkip) arrNew[arrNew.length] = arr[j];
        }
        this.antennas[i].frequencies = arrNew.toString();
        Settings("Position" + i) = this.antennas[i].frequencies;
    }
}
Ant.prototype.FindAntennaForSegment = function(segID) {
    var arr;
    var i,j;
    ACOMTools.RaiseEvent("DEBUG","Applicable antenna for SEG " + segID);
    for (i = 0; i < this.antennas.length; i++) {
        arr = this.antennas[i].frequencies.split(",");
        if (arr == null) continue;
        for (j = 0; j < arr.length; j++) {
            if (arr[j] == segID) return i;
        }
    }
    return -1;
}



//Packet_Operate
//    this.p = new ActiveXObject("newObjects.USPC.ACOMSerialProtocolPacket");

// Globals
var segments = new Array();
var Antennas = new Ant();
var g_mode = 0x8 | 0x10 | 0x20 | 0x0004 | 0x1000;
var g_SelfType = Settings.Option("SelfType");
var g_AmpType = 0x40;
var g_AntType = 0x50;
var g_TransType = 0x60;
var g_SelfGroup = Settings.Option("SelfGroup");
var LastSegment = -1;
var LastAntenna = -1;
var LastUserDef = -1;
var g_InitialRequestSend = false;
                        
var g_PROT = 0;
var g_HEATER = 0;

var g_SwitchingOn = -1;

// Initialization
// Init leds and so on
AmpPannel("ForwardPower").volume = 0;
AmpPannel("ReflectedPower").volume = 0;
//
AmpPannel("Prot").ui = false;
AmpPannel("Prot").Checked = false;
AmpPannel("Prot").focusIndication = false;
//
AmpPannel("Heater").ui = false;
AmpPannel("Heater").Checked = false;
AmpPannel("Heater").focusIndication = false;
//
AmpPannel("Plate").ui = false;
AmpPannel("Plate").Checked = false;
AmpPannel("Plate").focusIndication = false;
//
AmpPannel("Drive").ui = false;
AmpPannel("Drive").Checked = false;
AmpPannel("Drive").focusIndication = false;
//
AmpPannel("Opr").ui = false;
AmpPannel("Opr").Checked = false;
AmpPannel("Opr").focusIndication = true;
//
AmpPannel("Stby").ui = false;
AmpPannel("Stby").Checked = false;
AmpPannel("Stby").focusIndication = true;
//
AmpPannel("Comm").ui = false;
AmpPannel("Comm").Checked = false;
AmpPannel("Comm").focusIndication = false;
//
AmpPannel("Power").Text = "on/off";
AmpPannel("Power").ui = false;
AmpPannel("Power").Checked = false;
AmpPannel("Power").focusIndication = true;

AmpPannel.text("Text1") = "Frequency";
AmpPannel.text("Text2") = "Temperature";



// Init connection
var g_strConnectString = Settings("ConnectionString");
if (g_strConnectString.length > 0) {
    Driver.SetConnectString(g_strConnectString);
} else {
    // use default
    Driver.SetConnectString("COM1:BAUD=1200 PARITY=N DATA=8 STOP=1");
}


function ReviseLayout(layout) {
    switch (layout) {
        case 0:
            AmpPannel("ForwardPower").UseAmbientColors = false;
            AmpPannel("ForwardPower").border = 2;
            AmpPannel("ForwardPower").CaptionLabel = "Forward power (kW)";
            AmpPannel("ReflectedPower").UseAmbientColors = false;
            AmpPannel("ReflectedPower").border = 2;
            AmpPannel("ReflectedPower").CaptionLabel = "Reflected power (W)";
            AmpPannel("ForwardPower").BackColor = 0x00400040;
            AmpPannel("ForwardPower").ForeColor = 0xFFFFFF;
            AmpPannel("ReflectedPower").BackColor = 0x00400040;
            AmpPannel("ReflectedPower").ForeColor = 0xFFFFFF;
            AmpPannel("ReflectedPower").Labels=4;
        break;
        case 1:
        case 2:
        case 3:
        case 4:
            AmpPannel("ForwardPower").UseAmbientColors = true;
            AmpPannel("ForwardPower").border = 1;
            AmpPannel("ForwardPower").CaptionLabel = "";
            AmpPannel("ForwardPower").BackColor = ACOMTools.bgColor;
            AmpPannel("ForwardPower").ForeColor = ACOMTools.fgColor;
            AmpPannel("ReflectedPower").UseAmbientColors = true;
            AmpPannel("ReflectedPower").border = 1;
            AmpPannel("ReflectedPower").CaptionLabel = "";
            AmpPannel("ReflectedPower").BackColor = ACOMTools.bgColor;
            AmpPannel("ReflectedPower").ForeColor = ACOMTools.fgColor;
            AmpPannel("ReflectedPower").Labels=3;
        break;
    }
}


// Event handlers
function OnAmpSwitchOn(bConnected) {
    if (!bConnected) {
        ACOMTools.RaiseUIError(2,"Not connected");
    } else {
        // Turn on DTR/RTS signals
        var p = new ActiveXObject("newObjects.USPC.ACOMSerialProtocolPacket");
        p.packetType = 3;
        p.command = "SETRTS";
        Translator.Transmit(p);
        p.command = "SETDTR";
        Translator.Transmit(p);
        g_SwitchingOn = 0;
    }
}
function OnAmpSwitchOff(bConnected) {
    if (!bConnected) {
        ACOMTools.RaiseUIError(2,"Not connected");
    } else {
		var p = Packet(g_AmpType,g_SelfGroup,"0");
		ACOMTools.Transmit(p);        
    }
}
function OnAmpPannelLedClick(btn,name) {
    // btn false=left,true=right
	var p;
	if (!ACOMTools.IsConnected) return;
    switch (name) {
        case "Stby":
			p = Packet(g_AmpType,g_SelfGroup,"S");
			ACOMTools.Transmit(p);
        break;
		case "Opr":
			p = Packet(g_AmpType,g_SelfGroup,"O");
			ACOMTools.Transmit(p);
		break;
		case "Power":
		    if (Settings.Option("AmpSwitchOnOffEnabled") != 0) {
    		    if (g_nPackets) {
    		        // off request
    		        if (AmpPannel.askUser("Switch off the amplifier?",false) == 1) {
                        OnAmpSwitchOff(ACOMTools.IsConnected);
    		        }
    		    } else {
    		        // on request
    		        if (ACOMTools.IsConnected) {
        		        if (AmpPannel.askUser("Try to switch on the amplifier?",false) == 1) {
                            OnAmpSwitchOn(ACOMTools.IsConnected);
        		        }
        		    }
    		    }
    		}
		break;
    }
}
function OnAmpPannelLedState(bOn,name) {
    // bOn - On/Off
}
function OnAmpPannelForwardPowerClick(vol) {
    // Volume pressed

}
function OnAmpPannelReflectedPowerClick(vol) {
    // Volume pressed
    
}
function OnAntPannelPosition(pos) {
    // pos - antenna
    if (!Driver.CheckState(0x0003)) ACOMTools.RaiseUIError(1,"Not connected");
    var p = Packet(g_AmpType,g_SelfGroup,"W3");
    p.param(0) = pos;
    Translator.Transmit(p);
    ACOMTools.RaiseEvent("ANTENNA PANEL","Switching to position " + pos);

}
g_nTicks = 0;
g_nPackets = 0;
function OnTimer() {
    g_nTicks++;
    switch(g_PROT) {
        case 1:
            AmpPannel("Prot").checked = !AmpPannel("Prot").checked;
        break;
        case 2:
            AmpPannel("Prot").checked = true;
        break;
        default:
            AmpPannel("Prot").checked = false;
    }
    switch(g_HEATER) {
        case 1:
            AmpPannel("Heater").checked = !AmpPannel("Heater").checked;
        break;
        case 2:
            AmpPannel("Heater").checked = true;
        break;
        default:
            AmpPannel("Heater").checked = false;
    }
    if ( (g_nTicks % 20) == 0) {
        if (g_nPackets) {
            AmpPannel("Comm").checked = true;
        } else {
            AmpPannel("Comm").checked = false;
            AmpPannel("ForwardPower").volume = 0;
            AmpPannel("ReflectedPower").volume = 0;
	        ClearView();
        }
        g_nPackets = 0;
    }
    if (g_SwitchingOn >=0) {
        g_SwitchingOn++;
        if (g_SwitchingOn >= 3) {
            g_SwitchingOn = -1;
            // Clear the DTR/RTS
            var p = new ActiveXObject("newObjects.USPC.ACOMSerialProtocolPacket");
            p.packetType = 3;
            p.command = "CLRRTS";
            Translator.Transmit(p);
            p.command = "CLRDTR";
            Translator.Transmit(p);
        }
    }
                        
}
function OnPacketReceived(p) {
    var i;
    var seg;
    g_nPackets++;
    var p1;
    switch (p.packetType) {
        case 1: // Forward power
            AmpPannel("ForwardPower").volume = (2.4 * p.volume) / 16;
        break;
        case 2: // Reflection power
            AmpPannel("ReflectedPower").volume = (450 * p.volume) / 8;
        break;
        default:
            // Reflection 
                if (!p.isBroadcast) {
                    if (p.rxGroup != g_SelfGroup || p.rxType != g_SelfType) {
                        // Reflect
                        Translator.Transmit(p);
                        // Place for hooking packets for other devices
                        return;
                    }
                } else {
                    // Uncomment this to enable reflection of the broadcast packets
                    Translator.Transmit(p);
                }
                // Dispatch messages from the own group only
				if (p.txGroup != g_SelfGroup) return;
                switch (p.command) {
                    case "A":   // BAND Motor position (xxx,(L/D))
                    
                    break;
                    case "B":   // LOAD Motor position (xxx,(L/D))
                        
                    break;
                    case "C":   // TUNE Motor pos (xxx,(L/D))
                    break;
                    case "E":   // Error (xx,[yy])
						// nothing in the core    
                    break;
                    case "W5":  // OK - confirm
						// Not used currently
                    break;
                    case "H":   // Motors are in home position
                    
                    break;
                    case "I":   // Bars number temperature ??? (x)
                        AmpPannel("T").Text = "Temperature: " + p.param(0) * 10 + "[C]";
                    break;
                    case "L":   // Self address broadcast
		       	 AmpPannel("Power").checked = true;
		       	 if (!g_InitialRequestSend) {
		       	 	g_InitialRequestSend = true;
		       	 	p1 = DefAmpPacket("W7");
		       	 	ACOMTools.Transmit(p1);
		       	 }
                    break;
                    case "MM":  // RCU in measure
                    break;
                    case "O":   // Operate mode executed
                        AmpPannel("Opr").checked = true;
                        AmpPannel("Stby").checked = false;
                    break;
                    case "P":   // Input power marker is x
                    break;
                    case "Q":   // (x) 0 - DEF selected, 1 - USR sel, 2 - drive power unstable, 3 - end autotune proc
                        // Q - Q3 - handled by specific extensions
                    break;
                    case "S":   // standby executed
                        AmpPannel("Opr").checked = false;
                        AmpPannel("Stby").checked = true;
                    break;
                    case "U":   // misc led status
                        g_PROT = (p.param(0) & 0xC0) >> 6;
                        g_HEATER = (p.param(0) & 0x30) >> 4;
                        if (p.param(0) & 0x4) AmpPannel("Plate").checked = true;
                        else AmpPannel("Plate").checked = false;
                        if (p.param(0) & 0x2) AmpPannel("Drive").checked = true;
                        else AmpPannel("Drive").checked = false;
                        if (p.param(0) & 0x1) AmpPannel("Opr").checked = true;
                        else AmpPannel("Opr").checked = false;
                    break;
                    case "V":   // problems - 0-Noerror,1-stop drive power,2-drive pwr too high,3-loss of user data,4-batt warn
                        switch (p.param(0)) {
                            case 1:
                                AmpPannel("Status").Text = "Drive power stopped";
                                ACOMTools.PlaySound(2);
                            break;
                            case 2:
                                AmpPannel("Status").Text = "Drive power too high";
                                ACOMTools.PlaySound(2);
                            break; 
                            case 3:
                                AmpPannel("Status").Text = "User data lost";
                                ACOMTools.PlaySound(2);
                            break;
                            case 4:
                                AmpPannel("Status").Text = "Battery low";
                                ACOMTools.PlaySound(2);
                            break;
                        }
                    break;
                    case "W1":  // (xx-seg,a-antenna,u-0-DEF/1-USR,b-band ?)
                        AntPannel.position = p.param(1);
                        AmpPannel("Status").Text = "Antenna " + p.param(1) + " " + (p.param(2)?"USR":"DEF");
                        seg = GetSegmentByID(p.param(0));
                        if (seg != null) {
                            AmpPannel("Frequency").Text = seg.range + " " + (p.param(2)?"USR":"DEF");
                        }
                        LastSegment = p.param(0);
                        LastAntenna = p.param(1);
                        LastUserDef = p.param(2);
                    break;
                    case "W4":  // (a, u)
                        AntPannel.position = p.param(0);
                        LastAntenna = p.param(0);
                        AmpPannel("Status").Text = "Antenna " + p.param(0) + " " + (p.param(1)?"USR":"DEF");
                    break;
                    case "W6":  // (u)
                    	LastUserDef = p.param(0);
                    break;
                    case "0":   // power off 
                        AmpPannel("Opr").checked = false;
                        AmpPannel("Power").checked = false;
                        AmpPannel("Stby").checked = false;
                        AmpPannel("Status").Text = "Amplifier is shutting down";
                    break;
                    case "1":   // power on
                        AmpPannel("Power").checked = true;
                        AmpPannel("Status").Text = "Power is on";
                    break;
                    case "2":   // HV On
                        AmpPannel("Status").Text = "HV is on";
                    break;
                    case "3":   // Off but still cooling
                        AmpPannel("Stby").checked = false;
                        AmpPannel("Opr").checked = false;
                        AmpPannel("Status").Text = "Amplifier is shutting down";
                    break;
                    case "4":   // Off - complete
                        AmpPannel("Power").checked = false;
                        ClearView();
                    break;
                }
    }
}
function ClearView() {
    AmpPannel("ForwardPower").volume = 0;
    AmpPannel("ReflectedPower").volume = 0;
    AmpPannel("Prot").Checked = false;
    AmpPannel("Heater").Checked = false;
    AmpPannel("Plate").Checked = false;
    AmpPannel("Drive").Checked = false;
    AmpPannel("Opr").Checked = false;
    AmpPannel("Stby").Checked = false;
    AmpPannel("Comm").Checked = false;
    AmpPannel("Power").Checked = false;
    AmpPannel("Status").Text = "";
    AmpPannel("T").Text = "";
    AmpPannel("Frequency").Text = "";    
}

function Reconnect() {
    Driver.Close();
    Driver.SetConnectString(Settings("ConnectionString"));
    return true;

}


// Large definitions / resources
segments[segments.length] = {
    segment:  0,
    range:  "1.750 - 1.775 MHz",
    band:  "General Coverage",
    low:  1.75,
    high:  1.775
};
segments[segments.length] = {
    segment:  1,
    range:  "1.775 - 1.800 MHz",
    band:  "General Coverage",
    low:  1.775,
    high:  1.8
};
segments[segments.length] = {
    segment:  2,
    range:  "1.800 - 1.825 MHz",
    band:  "Amateur",
    low:  1.8,
    high:  1.825
};
segments[segments.length] = {
    segment:  3,
    range:  "1.825 - 1.850 MHz",
    band:  "Amateur",
    low:  1.825,
    high:  1.85
};
segments[segments.length] = {
    segment:  4,
    range:  "1.850 - 1.875 MHz",
    band:  "Amateur",
    low:  1.85,
    high:  1.875
};
segments[segments.length] = {
    segment:  5,
    range:  "1.875 - 1.900 MHz",
    band:  "Amateur",
    low:  1.875,
    high:  1.9
};
segments[segments.length] = {
    segment:  6,
    range:  "1.900 - 1.925 MHz",
    band:  "Amateur",
    low:  1.9,
    high:  1.925
};
segments[segments.length] = {
    segment:  7,
    range:  "1.925 - 1.950 MHz",
    band:  "Amateur",
    low:  1.925,
    high:  1.95
};
segments[segments.length] = {
    segment:  8,
    range:  "1.950 - 1.975 MHz",
    band:  "Amateur",
    low:  1.95,
    high:  1.975
};
segments[segments.length] = {
    segment:  9,
    range:  "1.975 - 2.000 MHz",
    band:  "Amateur",
    low:  1.975,
    high:  2.0
};
segments[segments.length] = {
    segment:  10,
    range:  "2.000 - 2.025 MHz",
    band:  "General Coverage",
    low:  2.0,
    high:  2.025
};
segments[segments.length] = {
    segment:  11,
    range:  "2.025 - 2.050 MHz",
    band:  "General Coverage",
    low:  2.025,
    high:  2.05
};
segments[segments.length] = {
    segment:  12,
    range:  "2.050 - 2.075 MHz",
    band:  "General Coverage",
    low:  2.05,
    high:  2.075
};
segments[segments.length] = {
    segment:  13,
    range:  "2.075 - 2.100 MHz",
    band:  "General Coverage",
    low:  2.075,
    high:  2.1
};
segments[segments.length] = {
    segment:  14,
    range:  "2.100 - 2.125 MHz",
    band:  "General Coverage",
    low:  2.1,
    high:  2.125
};
segments[segments.length] = {
    segment:  15,
    range:  "2.125 - 2.150 MHz",
    band:  "General Coverage",
    low:  2.125,
    high:  2.15
};
segments[segments.length] = {
    segment:  16,
    range:  "2.150 - 2.200 MHz",
    band:  "General Coverage",
    low:  2.15,
    high:  2.2
};
segments[segments.length] = {
    segment:  17,
    range:  "2.200 - 2.250 MHz",
    band:  "General Coverage",
    low:  2.2,
    high:  2.25
};
segments[segments.length] = {
    segment:  18,
    range:  "2.250 - 2.300 MHz",
    band:  "General Coverage",
    low:  2.25,
    high:  2.3
};
segments[segments.length] = {
    segment:  19,
    range:  "2.300 - 2.350 MHz",
    band:  "General Coverage",
    low:  2.3,
    high:  2.35
};
segments[segments.length] = {
    segment:  20,
    range:  "2.350 - 2.400 MHz",
    band:  "General Coverage",
    low:  2.35,
    high:  2.4
};
segments[segments.length] = {
    segment:  21,
    range:  "2.400 - 2.450 MHz",
    band:  "General Coverage",
    low:  2.4,
    high:  2.45
};
segments[segments.length] = {
    segment:  22,
    range:  "2.450 - 2.500 MHz",
    band:  "General Coverage",
    low:  2.45,
    high:  2.5
};
segments[segments.length] = {
    segment:  23,
    range:  "2.500 - 2.550 MHz",
    band:  "General Coverage",
    low:  2.5,
    high:  2.55
};
segments[segments.length] = {
    segment:  24,
    range:  "2.550 - 2.600 MHz",
    band:  "General Coverage",
    low:  2.55,
    high:  2.6
};
segments[segments.length] = {
    segment:  25,
    range:  "2.600 - 2.650 MHz",
    band:  "General Coverage",
    low:  2.6,
    high:  2.65
};
segments[segments.length] = {
    segment:  26,
    range:  "2.650 - 2.700 MHz",
    band:  "General Coverage",
    low:  2.65,
    high:  2.7
};
segments[segments.length] = {
    segment:  27,
    range:  "2.700 - 2.750 MHz",
    band:  "General Coverage",
    low:  2.7,
    high:  2.75
};
segments[segments.length] = {
    segment:  28,
    range:  "2.750 - 2.800 MHz",
    band:  "General Coverage",
    low:  2.75,
    high:  2.8
};
segments[segments.length] = {
    segment:  29,
    range:  "2.800 - 2.850 MHz",
    band:  "General Coverage",
    low:  2.8,
    high:  2.85
};
segments[segments.length] = {
    segment:  30,
    range:  "2.850 - 2.900 MHz",
    band:  "General Coverage",
    low:  2.85,
    high:  2.9
};
segments[segments.length] = {
    segment:  31,
    range:  "2.900 - 2.950 MHz",
    band:  "General Coverage",
    low:  2.9,
    high:  2.95
};
segments[segments.length] = {
    segment:  32,
    range:  "2.950 - 3.000 MHz",
    band:  "General Coverage",
    low:  2.95,
    high:  3.0
};
segments[segments.length] = {
    segment:  33,
    range:  "3.000 - 3.050 MHz",
    band:  "General Coverage",
    low:  3.0,
    high:  3.05
};
segments[segments.length] = {
    segment:  34,
    range:  "3.050 - 3.100 MHz",
    band:  "General Coverage",
    low:  3.05,
    high:  3.1
};
segments[segments.length] = {
    segment:  35,
    range:  "3.100 - 3.150 MHz",
    band:  "General Coverage",
    low:  3.1,
    high:  3.15
};
segments[segments.length] = {
    segment:  36,
    range:  "3.150 - 3.200 MHz",
    band:  "General Coverage",
    low:  3.15,
    high:  3.2
};
segments[segments.length] = {
    segment:  37,
    range:  "3.200 - 3.250 MHz",
    band:  "General Coverage",
    low:  3.2,
    high:  3.25
};
segments[segments.length] = {
    segment:  38,
    range:  "3.250 - 3.300 MHz",
    band:  "General Coverage",
    low:  3.25,
    high:  3.3
};
segments[segments.length] = {
    segment:  39,
    range:  "3.300 - 3.350 MHz",
    band:  "General Coverage",
    low:  3.3,
    high:  3.35
};
segments[segments.length] = {
    segment:  40,
    range:  "3.350 - 3.400 MHz",
    band:  "General Coverage",
    low:  3.35,
    high:  3.4
};
segments[segments.length] = {
    segment:  41,
    range:  "3.400 - 3.450 MHz",
    band:  "General Coverage",
    low:  3.4,
    high:  3.45
};
segments[segments.length] = {
    segment:  42,
    range:  "3.450 - 3.500 MHz",
    band:  "General Coverage",
    low:  3.45,
    high:  3.5
};
segments[segments.length] = {
    segment:  43,
    range:  "3.500 - 3.550 MHz",
    band:  "Amateur",
    low:  3.5,
    high:  3.55
};
segments[segments.length] = {
    segment:  44,
    range:  "3.550 - 3.600 MHz",
    band:  "Amateur",
    low:  3.55,
    high:  3.6
};
segments[segments.length] = {
    segment:  45,
    range:  "3.600 - 3.650 MHz",
    band:  "Amateur",
    low:  3.6,
    high:  3.65
};
segments[segments.length] = {
    segment:  46,
    range:  "3.650 - 3.700 MHz",
    band:  "Amateur",
    low:  3.65,
    high:  3.7
};
segments[segments.length] = {
    segment:  47,
    range:  "3.700 - 3.750 MHz",
    band:  "Amateur",
    low:  3.7,
    high:  3.75
};
segments[segments.length] = {
    segment:  48,
    range:  "3.750 - 3.800 MHz",
    band:  "Amateur",
    low:  3.75,
    high:  3.8
};
segments[segments.length] = {
    segment:  49,
    range:  "3.800 - 3.850 MHz",
    band:  "Amateur",
    low:  3.8,
    high:  3.85
};
segments[segments.length] = {
    segment:  50,
    range:  "3.850 - 3.900 MHz",
    band:  "Amateur",
    low:  3.85,
    high:  3.9
};
segments[segments.length] = {
    segment:  51,
    range:  "3.900 - 3.950 MHz",
    band:  "Amateur",
    low:  3.9,
    high:  3.95
};
segments[segments.length] = {
    segment:  52,
    range:  "3.950 - 4.000 MHz",
    band:  "Amateur",
    low:  3.95,
    high:  4.0
};
segments[segments.length] = {
    segment:  53,
    range:  "4.000 - 4.050 MHz",
    band:  "General Coverage",
    low:  4.0,
    high:  4.05
};
segments[segments.length] = {
    segment:  54,
    range:  "4.050 - 4.100 MHz",
    band:  "General Coverage",
    low:  4.05,
    high:  4.1
};
segments[segments.length] = {
    segment:  55,
    range:  "4.100 - 4.150 MHz",
    band:  "General Coverage",
    low:  4.1,
    high:  4.15
};
segments[segments.length] = {
    segment:  56,
    range:  "4.150 - 4.200 MHz",
    band:  "General Coverage",
    low:  4.15,
    high:  4.2
};
segments[segments.length] = {
    segment:  57,
    range:  "4.200 - 4.250 MHz",
    band:  "General Coverage",
    low:  4.2,
    high:  4.25
};
segments[segments.length] = {
    segment:  58,
    range:  "4.250 - 4.300 MHz",
    band:  "General Coverage",
    low:  4.25,
    high:  4.3
};
segments[segments.length] = {
    segment:  59,
    range:  "4.300 - 4.350 MHz",
    band:  "General Coverage",
    low:  4.3,
    high:  4.35
};
segments[segments.length] = {
    segment:  60,
    range:  "4.350 - 4.400 MHz",
    band:  "General Coverage",
    low:  4.35,
    high:  4.4
};
segments[segments.length] = {
    segment:  61,
    range:  "4.400 - 4.450 MHz",
    band:  "General Coverage",
    low:  4.4,
    high:  4.45
};
segments[segments.length] = {
    segment:  62,
    range:  "4.450 - 4.500 MHz",
    band:  "General Coverage",
    low:  4.45,
    high:  4.5
};
segments[segments.length] = {
    segment:  63,
    range:  "4.500 - 4.550 MHz",
    band:  "General Coverage",
    low:  4.5,
    high:  4.55
};
segments[segments.length] = {
    segment:  64,
    range:  "4.550 - 4.600 MHz",
    band:  "General Coverage",
    low:  4.55,
    high:  4.6
};
segments[segments.length] = {
    segment:  65,
    range:  "4.600 - 4.650 MHz",
    band:  "General Coverage",
    low:  4.6,
    high:  4.65
};
segments[segments.length] = {
    segment:  66,
    range:  "4.650 - 4.700 MHz",
    band:  "General Coverage",
    low:  4.65,
    high:  4.7
};
segments[segments.length] = {
    segment:  67,
    range:  "4.700 - 4.750 MHz",
    band:  "General Coverage",
    low:  4.7,
    high:  4.75
};
segments[segments.length] = {
    segment:  68,
    range:  "4.750 - 4.800 MHz",
    band:  "General Coverage",
    low:  4.75,
    high:  4.8
};
segments[segments.length] = {
    segment:  69,
    range:  "4.800 - 4.850 MHz",
    band:  "General Coverage",
    low:  4.8,
    high:  4.85
};
segments[segments.length] = {
    segment:  70,
    range:  "4.850 - 4.900 MHz",
    band:  "General Coverage",
    low:  4.85,
    high:  4.9
};
segments[segments.length] = {
    segment:  71,
    range:  "4.900 - 4.950 MHz",
    band:  "General Coverage",
    low:  4.9,
    high:  4.95
};
segments[segments.length] = {
    segment:  72,
    range:  "4.950 - 5.000 MHz",
    band:  "General Coverage",
    low:  4.95,
    high:  5.0
};
segments[segments.length] = {
    segment:  73,
    range:  "5.000 - 5.100 MHz",
    band:  "General Coverage",
    low:  5.0,
    high:  5.1
};
segments[segments.length] = {
    segment:  74,
    range:  "5.100 - 5.200 MHz",
    band:  "General Coverage",
    low:  5.1,
    high:  5.2
};
segments[segments.length] = {
    segment:  75,
    range:  "5.200 - 5.300 MHz",
    band:  "General Coverage",
    low:  5.2,
    high:  5.3
};
segments[segments.length] = {
    segment:  76,
    range:  "5.300 - 5.400 MHz",
    band:  "General Coverage",
    low:  5.3,
    high:  5.4
};
segments[segments.length] = {
    segment:  77,
    range:  "5.400 - 5.500 MHz",
    band:  "General Coverage",
    low:  5.4,
    high:  5.5
};
segments[segments.length] = {
    segment:  78,
    range:  "5.500 - 5.600 MHz",
    band:  "General Coverage",
    low:  5.5,
    high:  5.6
};
segments[segments.length] = {
    segment:  79,
    range:  "5.600 - 5.700 MHz",
    band:  "General Coverage",
    low:  5.6,
    high:  5.7
};
segments[segments.length] = {
    segment:  80,
    range:  "5.700 - 5.800 MHz",
    band:  "General Coverage",
    low:  5.7,
    high:  5.8
};
segments[segments.length] = {
    segment:  81,
    range:  "5.800 - 5.900 MHz",
    band:  "General Coverage",
    low:  5.8,
    high:  5.9
};
segments[segments.length] = {
    segment:  82,
    range:  "5.900 - 6.000 MHz",
    band:  "General Coverage",
    low:  5.9,
    high:  6.0
};
segments[segments.length] = {
    segment:  83,
    range:  "6.000 - 6.100 MHz",
    band:  "General Coverage",
    low:  6.0,
    high:  6.1
};
segments[segments.length] = {
    segment:  84,
    range:  "6.100 - 6.200 MHz",
    band:  "General Coverage",
    low:  6.1,
    high:  6.2
};
segments[segments.length] = {
    segment:  85,
    range:  "6.200 - 6.300 MHz",
    band:  "General Coverage",
    low:  6.2,
    high:  6.3
};
segments[segments.length] = {
    segment:  86,
    range:  "6.300 - 6.400 MHz",
    band:  "General Coverage",
    low:  6.3,
    high:  6.4
};
segments[segments.length] = {
    segment:  87,
    range:  "6.400 - 6.500 MHz",
    band:  "General Coverage",
    low:  6.4,
    high:  6.5
};
segments[segments.length] = {
    segment:  88,
    range:  "6.500 - 6.600 MHz",
    band:  "General Coverage",
    low:  6.5,
    high:  6.6
};
segments[segments.length] = {
    segment:  89,
    range:  "6.600 - 6.700 MHz",
    band:  "General Coverage",
    low:  6.6,
    high:  6.7
};
segments[segments.length] = {
    segment:  90,
    range:  "6.700 - 6.800 MHz",
    band:  "General Coverage",
    low:  6.7,
    high:  6.8
};
segments[segments.length] = {
    segment:  91,
    range:  "6.800 - 6.900 MHz",
    band:  "General Coverage",
    low:  6.8,
    high:  6.9
};
segments[segments.length] = {
    segment:  92,
    range:  "6.900 - 7.000 MHz",
    band:  "General Coverage",
    low:  6.9,
    high:  7.0
};
segments[segments.length] = {
    segment:  93,
    range:  "7.000 - 7.100 MHz",
    band:  "Amateur",
    low:  7.0,
    high:  7.1
};
segments[segments.length] = {
    segment:  94,
    range:  "7.100 - 7.200 MHz",
    band:  "Amateur",
    low:  7.1,
    high:  7.2
};
segments[segments.length] = {
    segment:  95,
    range:  "7.200 - 7.300 MHz",
    band:  "Amateur",
    low:  7.2,
    high:  7.3
};
segments[segments.length] = {
    segment:  96,
    range:  "7.300 - 7.400 MHz",
    band:  "General Coverage",
    low:  7.3,
    high:  7.4
};
segments[segments.length] = {
    segment:  97,
    range:  "7.400 - 7.500 MHz",
    band:  "General Coverage",
    low:  7.4,
    high:  7.5
};
segments[segments.length] = {
    segment:  98,
    range:  "7.500 - 7.600 MHz",
    band:  "General Coverage",
    low:  7.5,
    high:  7.6
};
segments[segments.length] = {
    segment:  99,
    range:  "7.600 - 7.700 MHz",
    band:  "General Coverage",
    low:  7.6,
    high:  7.7
};
segments[segments.length] = {
    segment:  100,
    range:  "7.700 - 7.800 MHz",
    band:  "General Coverage",
    low:  7.7,
    high:  7.8
};
segments[segments.length] = {
    segment:  101,
    range:  "7.800 - 7.900 MHz",
    band:  "General Coverage",
    low:  7.8,
    high:  7.9
};
segments[segments.length] = {
    segment:  102,
    range:  "7.900 - 8.000 MHz",
    band:  "General Coverage",
    low:  7.9,
    high:  8.0
};
segments[segments.length] = {
    segment:  103,
    range:  "8.000 - 8.100 MHz",
    band:  "General Coverage",
    low:  8.0,
    high:  8.1
};
segments[segments.length] = {
    segment:  104,
    range:  "8.100 - 8.200 MHz",
    band:  "General Coverage",
    low:  8.1,
    high:  8.2
};
segments[segments.length] = {
    segment:  105,
    range:  "8.200 - 8.300 MHz",
    band:  "General Coverage",
    low:  8.2,
    high:  8.3
};
segments[segments.length] = {
    segment:  106,
    range:  "8.300 - 8.400 MHz",
    band:  "General Coverage",
    low:  8.3,
    high:  8.4
};
segments[segments.length] = {
    segment:  107,
    range:  "8.400 - 8.500 MHz",
    band:  "General Coverage",
    low:  8.4,
    high:  8.5
};
segments[segments.length] = {
    segment:  108,
    range:  "8.500 - 8.600 MHz",
    band:  "General Coverage",
    low:  8.5,
    high:  8.6
};
segments[segments.length] = {
    segment:  109,
    range:  "8.600 - 8.700 MHz",
    band:  "General Coverage",
    low:  8.6,
    high:  8.7
};
segments[segments.length] = {
    segment:  110,
    range:  "8.700 - 8.800 MHz",
    band:  "General Coverage",
    low:  8.7,
    high:  8.8
};
segments[segments.length] = {
    segment:  111,
    range:  "8.800 - 8.900 MHz",
    band:  "General Coverage",
    low:  8.8,
    high:  8.9
};
segments[segments.length] = {
    segment:  112,
    range:  "8.900 - 9.000 MHz",
    band:  "General Coverage",
    low:  8.9,
    high:  9.0
};
segments[segments.length] = {
    segment:  113,
    range:  "9.000 - 9.100 MHz",
    band:  "General Coverage",
    low:  9.0,
    high:  9.1
};
segments[segments.length] = {
    segment:  114,
    range:  "9.100 - 9.200 MHz",
    band:  "General Coverage",
    low:  9.1,
    high:  9.2
};
segments[segments.length] = {
    segment:  115,
    range:  "9.200 - 9.300 MHz",
    band:  "General Coverage",
    low:  9.2,
    high:  9.3
};
segments[segments.length] = {
    segment:  116,
    range:  "9.300 - 9.400 MHz",
    band:  "General Coverage",
    low:  9.3,
    high:  9.4
};
segments[segments.length] = {
    segment:  117,
    range:  "9.400 - 9.500 MHz",
    band:  "General Coverage",
    low:  9.4,
    high:  9.5
};
segments[segments.length] = {
    segment:  118,
    range:  "9.500 - 9.600 MHz",
    band:  "General Coverage",
    low:  9.5,
    high:  9.6
};
segments[segments.length] = {
    segment:  119,
    range:  "9.600 - 9.700 MHz",
    band:  "General Coverage",
    low:  9.6,
    high:  9.7
};
segments[segments.length] = {
    segment:  120,
    range:  "9.700 - 9.800 MHz",
    band:  "General Coverage",
    low:  9.7,
    high:  9.8
};
segments[segments.length] = {
    segment:  121,
    range:  "9.800 - 9.900 MHz",
    band:  "General Coverage",
    low:  9.8,
    high:  9.9
};
segments[segments.length] = {
    segment:  122,
    range:  "9.900 - 10.000 MHz",
    band:  "General Coverage",
    low:  9.9,
    high:  10.0
};
segments[segments.length] = {
    segment:  123,
    range:  "10.000 - 10.100 MHz",
    band:  "Amateur",
    low:  10.0,
    high:  10.1
};
segments[segments.length] = {
    segment:  124,
    range:  "10.100 - 10.200 MHz",
    band:  "Amateur",
    low:  10.1,
    high:  10.2
};
segments[segments.length] = {
    segment:  125,
    range:  "10.200 - 10.300 MHz",
    band:  "General Coverage",
    low:  10.2,
    high:  10.3
};
segments[segments.length] = {
    segment:  126,
    range:  "10.300 - 10.400 MHz",
    band:  "General Coverage",
    low:  10.3,
    high:  10.4
};
segments[segments.length] = {
    segment:  127,
    range:  "10.400 - 10.500 MHz",
    band:  "General Coverage",
    low:  10.4,
    high:  10.5
};
segments[segments.length] = {
    segment:  128,
    range:  "10.500 - 10.600 MHz",
    band:  "General Coverage",
    low:  10.5,
    high:  10.6
};
segments[segments.length] = {
    segment:  129,
    range:  "10.600 - 10.700 MHz",
    band:  "General Coverage",
    low:  10.6,
    high:  10.7
};
segments[segments.length] = {
    segment:  130,
    range:  "10.700 - 10.800 MHz",
    band:  "General Coverage",
    low:  10.7,
    high:  10.8
};
segments[segments.length] = {
    segment:  131,
    range:  "10.800 - 10.900 MHz",
    band:  "General Coverage",
    low:  10.8,
    high:  10.9
};
segments[segments.length] = {
    segment:  132,
    range:  "10.900 - 11.000 MHz",
    band:  "General Coverage",
    low:  10.9,
    high:  11.0
};
segments[segments.length] = {
    segment:  133,
    range:  "11.000 - 11.100 MHz",
    band:  "General Coverage",
    low:  11.0,
    high:  11.1
};
segments[segments.length] = {
    segment:  134,
    range:  "11.100 - 11.200 MHz",
    band:  "General Coverage",
    low:  11.1,
    high:  11.2
};
segments[segments.length] = {
    segment:  135,
    range:  "11.200 - 11.300 MHz",
    band:  "General Coverage",
    low:  11.2,
    high:  11.3
};
segments[segments.length] = {
    segment:  136,
    range:  "11.300 - 11.400 MHz",
    band:  "General Coverage",
    low:  11.3,
    high:  11.4
};
segments[segments.length] = {
    segment:  137,
    range:  "11.400 - 11.500 MHz",
    band:  "General Coverage",
    low:  11.4,
    high:  11.5
};
segments[segments.length] = {
    segment:  138,
    range:  "11.500 - 11.600 MHz",
    band:  "General Coverage",
    low:  11.5,
    high:  11.6
};
segments[segments.length] = {
    segment:  139,
    range:  "11.600 - 11.700 MHz",
    band:  "General Coverage",
    low:  11.6,
    high:  11.7
};
segments[segments.length] = {
    segment:  140,
    range:  "11.700 - 11.800 MHz",
    band:  "General Coverage",
    low:  11.7,
    high:  11.8
};
segments[segments.length] = {
    segment:  141,
    range:  "11.800 - 11.900 MHz",
    band:  "General Coverage",
    low:  11.8,
    high:  11.9
};
segments[segments.length] = {
    segment:  142,
    range:  "11.900 - 12.000 MHz",
    band:  "General Coverage",
    low:  11.9,
    high:  12.0
};
segments[segments.length] = {
    segment:  143,
    range:  "12.000 - 12.100 MHz",
    band:  "General Coverage",
    low:  12.0,
    high:  12.1
};
segments[segments.length] = {
    segment:  144,
    range:  "12.100 - 12.200 MHz",
    band:  "General Coverage",
    low:  12.1,
    high:  12.2
};
segments[segments.length] = {
    segment:  145,
    range:  "12.200 - 12.300 MHz",
    band:  "General Coverage",
    low:  12.2,
    high:  12.3
};
segments[segments.length] = {
    segment:  146,
    range:  "12.300 - 12.400 MHz",
    band:  "General Coverage",
    low:  12.3,
    high:  12.4
};
segments[segments.length] = {
    segment:  147,
    range:  "12.400 - 12.500 MHz",
    band:  "General Coverage",
    low:  12.4,
    high:  12.5
};
segments[segments.length] = {
    segment:  148,
    range:  "12.500 - 12.600 MHz",
    band:  "General Coverage",
    low:  12.5,
    high:  12.6
};
segments[segments.length] = {
    segment:  149,
    range:  "12.600 - 12.700 MHz",
    band:  "General Coverage",
    low:  12.6,
    high:  12.7
};
segments[segments.length] = {
    segment:  150,
    range:  "12.700 - 12.800 MHz",
    band:  "General Coverage",
    low:  12.7,
    high:  12.8
};
segments[segments.length] = {
    segment:  151,
    range:  "12.800 - 12.900 MHz",
    band:  "General Coverage",
    low:  12.8,
    high:  12.9
};
segments[segments.length] = {
    segment:  152,
    range:  "12.900 - 13.000 MHz",
    band:  "General Coverage",
    low:  12.9,
    high:  13.0
};
segments[segments.length] = {
    segment:  153,
    range:  "13.000 - 13.100 MHz",
    band:  "General Coverage",
    low:  13.0,
    high:  13.1
};
segments[segments.length] = {
    segment:  154,
    range:  "13.100 - 13.200 MHz",
    band:  "General Coverage",
    low:  13.1,
    high:  13.2
};
segments[segments.length] = {
    segment:  155,
    range:  "13.200 - 13.300 MHz",
    band:  "General Coverage",
    low:  13.2,
    high:  13.3
};
segments[segments.length] = {
    segment:  156,
    range:  "13.300 - 13.400 MHz",
    band:  "General Coverage",
    low:  13.3,
    high:  13.4
};
segments[segments.length] = {
    segment:  157,
    range:  "13.400 - 13.500 MHz",
    band:  "General Coverage",
    low:  13.4,
    high:  13.5
};
segments[segments.length] = {
    segment:  158,
    range:  "13.500 - 13.600 MHz",
    band:  "General Coverage",
    low:  13.5,
    high:  13.6
};
segments[segments.length] = {
    segment:  159,
    range:  "13.600 - 13.700 MHz",
    band:  "General Coverage",
    low:  13.6,
    high:  13.7
};
segments[segments.length] = {
    segment:  160,
    range:  "13.700 - 13.800 MHz",
    band:  "General Coverage",
    low:  13.7,
    high:  13.8
};
segments[segments.length] = {
    segment:  161,
    range:  "13.800 - 13.900 MHz",
    band:  "General Coverage",
    low:  13.8,
    high:  13.9
};
segments[segments.length] = {
    segment:  162,
    range:  "13.900 - 14.000 MHz",
    band:  "General Coverage",
    low:  13.9,
    high:  14.0
};
segments[segments.length] = {
    segment:  163,
    range:  "14.000 - 14.100 MHz",
    band:  "Amateur",
    low:  14.0,
    high:  14.1
};
segments[segments.length] = {
    segment:  164,
    range:  "14.100 - 14.200 MHz",
    band:  "Amateur",
    low:  14.1,
    high:  14.2
};
segments[segments.length] = {
    segment:  165,
    range:  "14.200 - 14.300 MHz",
    band:  "Amateur",
    low:  14.2,
    high:  14.3
};
segments[segments.length] = {
    segment:  166,
    range:  "14.300 - 14.400 MHz",
    band:  "Amateur",
    low:  14.3,
    high:  14.4
};
segments[segments.length] = {
    segment:  167,
    range:  "14.400 - 14.500 MHz",
    band:  "General Coverage",
    low:  14.4,
    high:  14.5
};
segments[segments.length] = {
    segment:  168,
    range:  "14.500 - 14.600 MHz",
    band:  "General Coverage",
    low:  14.5,
    high:  14.6
};
segments[segments.length] = {
    segment:  169,
    range:  "14.600 - 14.700 MHz",
    band:  "General Coverage",
    low:  14.6,
    high:  14.7
};
segments[segments.length] = {
    segment:  170,
    range:  "14.700 - 14.800 MHz",
    band:  "General Coverage",
    low:  14.7,
    high:  14.8
};
segments[segments.length] = {
    segment:  171,
    range:  "14.800 - 14.900 MHz",
    band:  "General Coverage",
    low:  14.8,
    high:  14.9
};
segments[segments.length] = {
    segment:  172,
    range:  "14.900 - 15.000 MHz",
    band:  "General Coverage",
    low:  14.9,
    high:  15.0
};
segments[segments.length] = {
    segment:  173,
    range:  "15.000 - 15.150 MHz",
    band:  "General Coverage",
    low:  15.0,
    high:  15.15
};
segments[segments.length] = {
    segment:  174,
    range:  "15.150 - 15.300 MHz",
    band:  "General Coverage",
    low:  15.15,
    high:  15.3
};
segments[segments.length] = {
    segment:  175,
    range:  "15.300 - 15.450 MHz",
    band:  "General Coverage",
    low:  15.3,
    high:  15.45
};
segments[segments.length] = {
    segment:  176,
    range:  "15.450 - 15.600 MHz",
    band:  "General Coverage",
    low:  15.45,
    high:  15.6
};
segments[segments.length] = {
    segment:  177,
    range:  "15.600 - 15.750 MHz",
    band:  "General Coverage",
    low:  15.6,
    high:  15.75
};
segments[segments.length] = {
    segment:  178,
    range:  "15.750 - 15.900 MHz",
    band:  "General Coverage",
    low:  15.75,
    high:  15.9
};
segments[segments.length] = {
    segment:  179,
    range:  "15.900 - 16.050 MHz",
    band:  "General Coverage",
    low:  15.9,
    high:  16.05
};
segments[segments.length] = {
    segment:  180,
    range:  "16.050 - 16.200 MHz",
    band:  "General Coverage",
    low:  16.05,
    high:  16.2
};
segments[segments.length] = {
    segment:  181,
    range:  "16.200 - 16.350 MHz",
    band:  "General Coverage",
    low:  16.2,
    high:  16.35
};
segments[segments.length] = {
    segment:  182,
    range:  "16.350 - 16.500 MHz",
    band:  "General Coverage",
    low:  16.35,
    high:  16.5
};
segments[segments.length] = {
    segment:  183,
    range:  "16.500 - 16.650 MHz",
    band:  "General Coverage",
    low:  16.5,
    high:  16.65
};
segments[segments.length] = {
    segment:  184,
    range:  "16.650 - 16.800 MHz",
    band:  "General Coverage",
    low:  16.65,
    high:  16.8
};
segments[segments.length] = {
    segment:  185,
    range:  "16.800 - 16.950 MHz",
    band:  "General Coverage",
    low:  16.8,
    high:  16.95
};
segments[segments.length] = {
    segment:  186,
    range:  "16.950 - 17.100 MHz",
    band:  "General Coverage",
    low:  16.95,
    high:  17.1
};
segments[segments.length] = {
    segment:  187,
    range:  "17.100 - 17.250 MHz",
    band:  "General Coverage",
    low:  17.1,
    high:  17.25
};
segments[segments.length] = {
    segment:  188,
    range:  "17.250 - 17.400 MHz",
    band:  "General Coverage",
    low:  17.25,
    high:  17.4
};
segments[segments.length] = {
    segment:  189,
    range:  "17.400 - 17.550 MHz",
    band:  "General Coverage",
    low:  17.4,
    high:  17.55
};
segments[segments.length] = {
    segment:  190,
    range:  "17.550 - 17.700 MHz",
    band:  "General Coverage",
    low:  17.55,
    high:  17.7
};
segments[segments.length] = {
    segment:  191,
    range:  "17.700 - 17.850 MHz",
    band:  "General Coverage",
    low:  17.7,
    high:  17.85
};
segments[segments.length] = {
    segment:  192,
    range:  "17.850 - 18.000 MHz",
    band:  "General Coverage",
    low:  17.85,
    high:  18.0
};
segments[segments.length] = {
    segment:  193,
    range:  "18.000 - 18.150 MHz",
    band:  "Amateur",
    low:  18.0,
    high:  18.15
};
segments[segments.length] = {
    segment:  194,
    range:  "18.150 - 18.300 MHz",
    band:  "Amateur",
    low:  18.15,
    high:  18.3
};
segments[segments.length] = {
    segment:  195,
    range:  "18.300 - 18.450 MHz",
    band:  "General Coverage",
    low:  18.3,
    high:  18.45
};
segments[segments.length] = {
    segment:  196,
    range:  "18.450 - 18.600 MHz",
    band:  "General Coverage",
    low:  18.45,
    high:  18.6
};
segments[segments.length] = {
    segment:  197,
    range:  "18.600 - 18.750 MHz",
    band:  "General Coverage",
    low:  18.6,
    high:  18.75
};
segments[segments.length] = {
    segment:  198,
    range:  "18.750 - 18.900 MHz",
    band:  "General Coverage",
    low:  18.75,
    high:  18.9
};
segments[segments.length] = {
    segment:  199,
    range:  "18.900 - 19.050 MHz",
    band:  "General Coverage",
    low:  18.9,
    high:  19.05
};
segments[segments.length] = {
    segment:  200,
    range:  "19.050 - 19.200 MHz",
    band:  "General Coverage",
    low:  19.05,
    high:  19.2
};
segments[segments.length] = {
    segment:  201,
    range:  "19.200 - 19.350 MHz",
    band:  "General Coverage",
    low:  19.2,
    high:  19.35
};
segments[segments.length] = {
    segment:  202,
    range:  "19.350 - 19.500 MHz",
    band:  "General Coverage",
    low:  19.35,
    high:  19.5
};
segments[segments.length] = {
    segment:  203,
    range:  "19.500 - 19.650 MHz",
    band:  "General Coverage",
    low:  19.5,
    high:  19.65
};
segments[segments.length] = {
    segment:  204,
    range:  "19.650 - 19.800 MHz",
    band:  "General Coverage",
    low:  19.65,
    high:  19.8
};
segments[segments.length] = {
    segment:  205,
    range:  "19.800 - 19.950 MHz",
    band:  "General Coverage",
    low:  19.8,
    high:  19.95
};
segments[segments.length] = {
    segment:  206,
    range:  "19.950 - 20.100 MHz",
    band:  "General Coverage",
    low:  19.95,
    high:  20.1
};
segments[segments.length] = {
    segment:  207,
    range:  "20.100 - 20.250 MHz",
    band:  "General Coverage",
    low:  20.1,
    high:  20.25
};
segments[segments.length] = {
    segment:  208,
    range:  "20.250 - 20.400 MHz",
    band:  "General Coverage",
    low:  20.25,
    high:  20.4
};
segments[segments.length] = {
    segment:  209,
    range:  "20.400 - 20.550 MHz",
    band:  "General Coverage",
    low:  20.4,
    high:  20.55
};
segments[segments.length] = {
    segment:  210,
    range:  "20.550 - 20.700 MHz",
    band:  "General Coverage",
    low:  20.55,
    high:  20.7
};
segments[segments.length] = {
    segment:  211,
    range:  "20.700 - 20.850 MHz",
    band:  "General Coverage",
    low:  20.7,
    high:  20.85
};
segments[segments.length] = {
    segment:  212,
    range:  "20.850 - 21.000 MHz",
    band:  "General Coverage",
    low:  20.85,
    high:  21.0
};
segments[segments.length] = {
    segment:  213,
    range:  "21.000 - 21.150 MHz",
    band:  "Amateur",
    low:  21.0,
    high:  21.15
};
segments[segments.length] = {
    segment:  214,
    range:  "21.150 - 21.300 MHz",
    band:  "Amateur",
    low:  21.15,
    high:  21.3
};
segments[segments.length] = {
    segment:  215,
    range:  "21.300 - 21.450 MHz",
    band:  "Amateur",
    low:  21.3,
    high:  21.45
};
segments[segments.length] = {
    segment:  216,
    range:  "21.450 - 21.600 MHz",
    band:  "General Coverage",
    low:  21.45,
    high:  21.6
};
segments[segments.length] = {
    segment:  217,
    range:  "21.600 - 21.750 MHz",
    band:  "General Coverage",
    low:  21.6,
    high:  21.75
};
segments[segments.length] = {
    segment:  218,
    range:  "21.750 - 21.900 MHz",
    band:  "General Coverage",
    low:  21.75,
    high:  21.9
};
segments[segments.length] = {
    segment:  219,
    range:  "21.900 - 22.100 MHz",
    band:  "General Coverage",
    low:  21.9,
    high:  22.1
};
segments[segments.length] = {
    segment:  220,
    range:  "22.100 - 22.300 MHz",
    band:  "General Coverage",
    low:  22.1,
    high:  22.3
};
segments[segments.length] = {
    segment:  221,
    range:  "22.300 - 22.500 MHz",
    band:  "General Coverage",
    low:  22.3,
    high:  22.5
};
segments[segments.length] = {
    segment:  222,
    range:  "22.500 - 22.750 MHz",
    band:  "General Coverage",
    low:  22.5,
    high:  22.75
};
segments[segments.length] = {
    segment:  223,
    range:  "22.750 - 23.000 MHz",
    band:  "General Coverage",
    low:  22.75,
    high:  23.0
};
segments[segments.length] = {
    segment:  224,
    range:  "23.000 - 23.250 MHz",
    band:  "General Coverage",
    low:  23.0,
    high:  23.25
};
segments[segments.length] = {
    segment:  225,
    range:  "23.250 - 23.500 MHz",
    band:  "General Coverage",
    low:  23.25,
    high:  23.5
};
segments[segments.length] = {
    segment:  226,
    range:  "23.500 - 23.750 MHz",
    band:  "General Coverage",
    low:  23.5,
    high:  23.75
};
segments[segments.length] = {
    segment:  227,
    range:  "23.750 - 24.000 MHz",
    band:  "General Coverage",
    low:  23.75,
    high:  24.0
};
segments[segments.length] = {
    segment:  228,
    range:  "24.000 - 24.250 MHz",
    band:  "General Coverage",
    low:  24.0,
    high:  24.25
};
segments[segments.length] = {
    segment:  229,
    range:  "24.250 - 24.500 MHz",
    band:  "General Coverage",
    low:  24.25,
    high:  24.5
};
segments[segments.length] = {
    segment:  230,
    range:  "24.500 - 24.750 MHz",
    band:  "24.5/28MHz Licensed",
    low:  24.5,
    high:  24.75
};
segments[segments.length] = {
    segment:  231,
    range:  "24.750 - 25.000 MHz",
    band:  "24.5/28MHz Licensed",
    low:  24.75,
    high:  25.0
};
segments[segments.length] = {
    segment:  232,
    range:  "25.000 - 25.250 MHz",
    band:  "General Coverage",
    low:  25.0,
    high:  25.25
};
segments[segments.length] = {
    segment:  233,
    range:  "25.250 - 25.500 MHz",
    band:  "General Coverage",
    low:  25.25,
    high:  25.5
};
segments[segments.length] = {
    segment:  234,
    range:  "25.500 - 25.750 MHz",
    band:  "General Coverage",
    low:  25.5,
    high:  25.75
};
segments[segments.length] = {
    segment:  235,
    range:  "25.750 - 26.000 MHz",
    band:  "General Coverage",
    low:  25.75,
    high:  26.0
};
segments[segments.length] = {
    segment:  236,
    range:  "26.000 - 26.250 MHz",
    band:  "General Coverage",
    low:  26.0,
    high:  26.25
};
segments[segments.length] = {
    segment:  237,
    range:  "26.250 - 26.500 MHz",
    band:  "General Coverage",
    low:  26.25,
    high:  26.5
};
segments[segments.length] = {
    segment:  238,
    range:  "26.500 - 26.800 MHz",
    band:  "General Coverage",
    low:  26.5,
    high:  26.8
};
segments[segments.length] = {
    segment:  239,
    range:  "26.800 - 27.100 MHz",
    band:  "General Coverage",
    low:  26.8,
    high:  27.1
};
segments[segments.length] = {
    segment:  240,
    range:  "27.100 - 27.400 MHz",
    band:  "General Coverage",
    low:  27.1,
    high:  27.4
};
segments[segments.length] = {
    segment:  241,
    range:  "27.400 - 27.700 MHz",
    band:  "General Coverage",
    low:  27.4,
    high:  27.7
};
segments[segments.length] = {
    segment:  242,
    range:  "27.700 - 28.000 MHz",
    band:  "General Coverage",
    low:  27.7,
    high:  28.0
};
segments[segments.length] = {
    segment:  243,
    range:  "28.000 - 28.300 MHz",
    band:  "24.5/28MHz Licensed",
    low:  28.0,
    high:  28.3
};
segments[segments.length] = {
    segment:  244,
    range:  "28.300 - 28.600 MHz",
    band:  "24.5/28MHz Licensed",
    low:  28.3,
    high:  28.6
};
segments[segments.length] = {
    segment:  245,
    range:  "28.600 - 28.900 MHz",
    band:  "24.5/28MHz Licensed",
    low:  28.6,
    high:  28.9
};
segments[segments.length] = {
    segment:  246,
    range:  "28.900 - 29.200 MHz",
    band:  "24.5/28MHz Licensed",
    low:  28.9,
    high:  29.2
};
segments[segments.length] = {
    segment:  247,
    range:  "29.200 - 29.500 MHz",
    band:  "24.5/28MHz Licensed",
    low:  29.2,
    high:  29.5
};
segments[segments.length] = {
    segment:  248,
    range:  "29.500 - 29.800 MHz",
    band:  "24.5/28MHz Licensed",
    low:  29.5,
    high:  29.8
};
segments[segments.length] = {
    segment:  249,
    range:  "29.800 - 30.000 MHz",
    band:  "General Coverage",
    low:  29.8,
    high:  30.0
};

